package ga.core.individual.population;

import ga.core.GA;
import ga.core.algorithm.util.ClusterUtil;
import ga.core.algorithm.util.RandomSingleton;
import ga.core.evaluation.IFitnessEvaluator;
import ga.core.individual.IClusterableIndividual;
import ga.core.individual.IDebugInfo;
import ga.core.individual.IIndividualFactory;
import ga.core.individual.IndividualList;
import ga.core.validation.GAContext;
import ga.core.validation.IValidator;

import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.commons.math.stat.clustering.Cluster;
import org.apache.commons.math.stat.clustering.KMeansPlusPlusClusterer;

/**
 * Implementation of a clusterable population that uses the
 * {@link KMeansPlusPlusClusterer} of Apache Commons Math.
 * 
 * @param <T>
 *          The generic type of individuals.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public class KMeansClusterPopulation<T extends IClusterableIndividual<T>>
    implements IClusterPopulation<T> {

  // the logger for this class
  private static final Logger LOGGER = Logger
      .getLogger(KMeansClusterPopulation.class.getName());

  private static final int MAX_ITERATIONS = 100;

  private final IndividualList<T> pop = new IndividualList<T>();
  private final IIndividualFactory<T> factory;
  private IFitnessEvaluator<T> evaluator;

  private boolean allowDuplicates;

  private final Random rnd = RandomSingleton.getRandom();

  private int initIndividualCount = 100;
  private int clusterCount = 10;

  private List<Cluster<T>> clusters;

  private final KMeansPlusPlusClusterer<T> clusterer;

  /**
   * Creates a new clusterable population.
   * 
   * @param factory
   *          The individual factory.
   * @param initIndividualCount
   *          Init size of the population.
   * @param clusterCount
   *          Number of clusters to divide the population into.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public KMeansClusterPopulation(final IIndividualFactory<T> factory,
      final int initIndividualCount, final int clusterCount) {
    this(factory, initIndividualCount, clusterCount, true);
  }

  /**
   * Creates a new clusterable population.
   * 
   * @param factory
   *          The individual factory.
   * @param initIndividualCount
   *          Init size of the population.
   * @param clusterCount
   *          Number of clusters to divide the population into.
   * @param allowDuplicates
   *          Allow duplicate individuals if <code>true</code> but prevent them
   *          otherwise.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public KMeansClusterPopulation(final IIndividualFactory<T> factory,
      final int initIndividualCount, final int clusterCount,
      final boolean allowDuplicates) {
    this.factory = factory;
    this.initIndividualCount = initIndividualCount;
    this.clusterCount = clusterCount;
    this.allowDuplicates = allowDuplicates;

    clusterer = new KMeansPlusPlusClusterer<T>(RandomSingleton.getRandom());
  }

  @Override
  public void setEvaluator(final IFitnessEvaluator<T> evaluator) {
    this.evaluator = evaluator;
  }

  @Override
  public void initRandomly(final IValidator<T> validator,
      final GAContext context) {

    try {
      @SuppressWarnings("unchecked")
      final T template = (T) context.get(GA.KEY_INIT_INDIVIDUAL);

      if (template != null && validator.isValid(template, context)) {
        final int percentage = context.getInt(
            GA.KEY_INIT_INDIVIDUAL_PERCENTAGE, 1);

        final int count = Math.min(
            Math.max((percentage * initIndividualCount) / 100, 1), 100);

        for (int i = 0; i < count; i++) {
          pop.add(template);
        }
      }
    } catch (final Exception e) {
      e.printStackTrace();
    }

    while (pop.size() < initIndividualCount) {
      final T ind = factory.newIndividual(context);

      do {
        ind.initRandomly();
      } while (validator != null && !validator.isValid(ind, context));

      pop.add(ind);
    }

    doClustering();
  }

  @Override
  public IndividualList<T> getIndividuals() {
    return pop;
  }

  @Override
  public void addIndividuals(final T... individuals) {
    for (final T ind : individuals) {
      if (allowDuplicates || !pop.contains(ind)) {
        pop.add(ind);
      }
    }
  }

  @Override
  public void addIndividual(final T individual) {
    if (allowDuplicates || !pop.contains(individual)) {
      pop.add(individual);
    }
  }

  @Override
  public void addIndividuals(final IndividualList<T> individuals) {
    for (final T ind : individuals) {
      if (allowDuplicates || !pop.contains(ind)) {
        pop.add(ind);
      }
    }
  }

  @Override
  public void clear() {
    pop.clear();

    clusters.clear();
  }

  @Override
  public void evaluateAutomatic() {
    if (clusters == null || clusters.isEmpty()) {
      doClustering();
    }

    for (final Cluster<T> c : clusters) {
      final T ind = c.getCenter();
      if (!ind.isEvaluated()) {
        evaluator.evaluate(ind);
      }
    }

    assignFitness();
  }

  @Override
  public String toString() {
    final StringBuilder sb = new StringBuilder();

    for (final T ind : pop) {
      sb.append(ind);
      sb.append('\n');
    }

    if (sb.length() > 0) {
      sb.setLength(sb.length() - 1);
    }

    return sb.toString();
  }

  @Override
  public int size() {
    return pop.size();
  }

  @Override
  public T getUnfittestIndividual() {
    if (pop.size() == 0) {
      return null;
    }

    pop.sort(false);
    return pop.get(0);
  }

  @Override
  public T getFittestIndividual() {
    if (pop.size() == 0) {
      return null;
    }

    pop.sort(true);
    return pop.get(0);
  }

  @Override
  public T getEliteIndividual() {
    // get elite only from direct evaluated individuals
    T ind = clusters.get(0).getCenter();

    for (final Cluster<T> c : clusters) {
      final T t = c.getCenter();
      if (t.getFitness() > ind.getFitness()) {
        ind = t;
      }
    }
    return ind;
  }

  @Override
  public T getRandomIndividualForEvaluation() {
    return clusters.get(rnd.nextInt(clusters.size())).getCenter();
  }

  @Override
  public T getRandomIndividualForSelection() {
    return pop.get(rnd.nextInt(pop.size()));
  }

  @Override
  public IndividualList<T> getUnevaluatedIndividuals() {
    final IndividualList<T> list = new IndividualList<T>();
    for (final Cluster<T> c : clusters) {
      final T ind = c.getCenter();
      if (!ind.isEvaluated()) {
        list.add(ind);
      }
    }

    return list;
  }

  /**
   * Method that creates a string representation of the population, usually for
   * debugging purposes.
   * 
   * @return The population as string.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public String toClusterString() {
    if (clusters == null || clusters.isEmpty()) {
      return "";
    }

    final StringBuilder sb = new StringBuilder();

    for (final Cluster<T> c : clusters) {
      final boolean first = true;

      final T center = c.getCenter();

      if (center instanceof IDebugInfo) {
        sb.append(center.getId() + ": "
            + ((IDebugInfo) center).getGenotypeString());
      } else {
        sb.append(center.toString());
      }

      for (final T ind : c.getPoints()) {
        if (!ind.equals(center)) {
          sb.append(", ");
          if (ind instanceof IDebugInfo) {
            sb.append(ind.getId() + ": "
                + ((IDebugInfo) ind).getGenotypeString());
          } else {
            sb.append(ind.toString());
          }
        }
      }
      sb.append('\n');
    }

    return sb.toString();
  }

  @Override
  public boolean isEmpty() {
    return pop.isEmpty();
  }

  @Override
  public boolean isAllowDuplicates() {
    return allowDuplicates;
  }

  @Override
  public int getEvaluatedIndividualCount() {
    int i = 0;
    for (final T ind : pop) {
      if (ind.isEvaluated()) {
        i++;
      }
    }

    return i;
  }

  @Override
  public boolean containsAny(final IndividualList<T> list) {
    for (final T ind : list) {
      if (pop.contains(ind)) {
        return true;
      }
    }

    return false;
  }

  @Override
  public Iterator<T> iterator() {
    return pop.iterator();
  }

  @Override
  public int getInitIndividualCount() {
    return initIndividualCount;
  }

  @Override
  public void setInitIndividualCount(final int individualCount) {
    this.initIndividualCount = individualCount;
  }

  @Override
  public void assignFitness(final T ind) {
    ClusterUtil.assignFitness(clusters, ind);
  }

  @Override
  public void assignFitness() {
    for (final Cluster<T> c : clusters) {
      assignFitness(c.getCenter());
    }
  }

  @Override
  public void doClustering() {

    try {
      clusters = clusterer.cluster(pop, clusterCount, MAX_ITERATIONS);
    } catch (final Exception e) {
      LOGGER.log(Level.SEVERE,
          "Clustering failed. Population size: " + pop.size()
              + ", cluster count: " + clusterCount + ", Max iterations: "
              + MAX_ITERATIONS, e);
    }
  }
}
